#ifndef __CXMLNode__
#define __CXMLNode__

#include "../Collections/TCountedPointerArray.hpp"
#include "CXMLAttribute.hpp"
using Exponent::Collections::TCountedPointerArray;
using Exponent::IO::CXMLAttribute;

namespace Exponent
{
	namespace IO
	{
		/**
		 * @class CXMLNode CXMLNode.hpp
		 * @brief Represents an xml node
		 *
		 * @date 17/08/2005
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: CXMLNode.hpp,v 1.3 2007/02/08 21:06:44 paul Exp $
		 */
		class CXMLNode : public CCountedObject
		{
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */

//	===========================================================================

		public:

//	===========================================================================
		
			/**
			 * Construction
			 * @param name The name of the node
			 * @param parent The parent node
			 */
			CXMLNode(const CString &name, CXMLNode *parent = NULL);

			/**
			 * Destruction
			 */
			virtual ~CXMLNode();

//	===========================================================================

			/**
			 * Set parent node
			 * @param parent The parent node
			 */
			void setParentNode(CXMLNode *parent);

			/**
			 * Get the parent node
			 * @retval CXMLNode *The parent node
			 */
			CXMLNode *getParentNode();

			/**
			 * SEt the name of the node
			 * @param name The name of the node
			 */
			void setNodeName(const CString &name);

			/**
			 * Get the name of the node
			 * @retval const CString &name
			 */
			const CString &getNodeName() const;

//	===========================================================================
			
			/**
			 * Add an attribute to the node
			 * @param name The name of the attribute
			 * @param value The value of the attribute
			 */
			void addAttribute(const CString &name, const CString &value);

			/**
			 * Add a new attribute
			 * @param attribute The new attribute
			 */
			void addAttribute(CXMLAttribute *attribute);

			/**
			 * Get the number of attributes
			 * @retval long Number of attributes
			 */
			long getNumberOfAttributes() const;

			/**
			 * Get an attribute at an index
			 * @param index the index of the attribute
			 * @retval CXMLAttribute* The attribute from the index or NULL on error
			 */
			CXMLAttribute *getAttribute(const long index);

			/**
			 * Get an attribute wiht the name specified
			 * @param name The name of the attribute
			 * @retval CXMLAttribute* The attribute with the nae or NULL on error
			 */
			CXMLAttribute *getAttribute(const CString &name);

			/**
			 * Get an attribute at an index
			 * @param index the index of the attribute
			 * @retval CXMLAttribute* The attribute from the index or NULL on error
			 */
			const CXMLAttribute *getConstAttribute(const long index) const;

			/**
			 * Get an attribute wiht the name specified
			 * @param name The name of the attribute
			 * @retval const CXMLAttribute* The attribute with the nae or NULL on error
			 */
			const CXMLAttribute *getConstAttribute(const CString &name) const;

			/**
			 * Value of attribute named a specific thing
			 * @retval CString The value of the attribute, on error is CString::CSTRING_EMPTYSTRING;
			 */
			CString getValueOfAttributeNamed(const CString &name);

			/**
			 * Remove an attribute at a specified index
			 * @param attribute The attributte to remove
			 */
			void removeAttribute(CXMLAttribute *attribute);

			/**
			 * Remove the named attribute
			 * @param name The name of the attribute
			 */
			void removeAttribute(const CString &name);

			/**
			 * Clear all the attributes
			 */
			void clearAttributes();

//	===========================================================================

			/**
			 * Add a child node
			 * @param node The child node to add
			 */
			void addChildNode(CXMLNode *node);

			/**
			 * Add a child node
			 * @param name The name of the child node
			 */
			void addChildNode(const CString &name);

			/**
			 * Get the number of chid nodes
			 * @retval long The number of child nodes
			 */
			long getNumberOfChildNodes() const;

			/**
			 * Get an node at an index
			 * @param index the index of the node
			 * @retval const CXMLNode* The child node or NULL on error
			 */
			const CXMLNode *getConstChildNode(const long index) const;

			/**
			 * Get an node wiht the name specified
			 * @param name The name of the node
			 * @retval const CXMLNode* The child node or NULL on error
			 */
			const CXMLNode *getConstChildNode(const CString &name) const;

			/**
			 * Get the first chld node with the name specified, depth first search
			 * @param name THe name of the node
			 * @retval const CXMLNode* The first child node or NULL on error
			 */
			const CXMLNode *getConstFirstChildNode(const CString &name) const;

			/**
			 * Get an node at an index
			 * @param index the index of the node
			 * @retval CXMLNode* The child node or NULL on error
			 */
			CXMLNode *getChildNode(const long index);

			/**
			 * Get an node wiht the name specified
			 * @param name The name of the node
			 * @retval CXMLNode* The child node or NULL on error
			 */
			CXMLNode *getChildNode(const CString &name);

			/**
			 * Get the first chld node with the name specified, depth first search
			 * @param name THe name of the node
			 * @retval CXMLNode* The first child node or NULL on error
			 */
			CXMLNode *getFirstChildNode(const CString &name);

			/**
			 * Remove an node at a specified index
			 * @param node The node to remove
			 */
			void removeChildNode(CXMLNode *node);

			/**
			 * Remove the named node
			 * @param name The name of the node
			 */
			void removeChildNode(const CString &name);

			/**
			 * Clear all the attributes
			 */
			void clearChildNodes();

//	===========================================================================
			
		protected:

//	===========================================================================
		
			CString m_name;											/**< Name of the node */
			TCountedPointerArray<CXMLAttribute> m_attributes;		/**< Tag attributes */
			CXMLNode *m_parentNode;									/**< Parent node */
			TCountedPointerArray<CXMLNode> m_children;				/**< Children nodes */
		};
	}
}

#endif		// End of CXMLNode.hpp